/**
* \file: automounter_api.h
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: automounter
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#ifndef AUTOMOUNTER_API_H_
#define AUTOMOUNTER_API_H_

#ifdef __cplusplus
extern "C" {
#endif

#include "automounter_types.h"

/**
 * \enum automounter_api_state_t
 * \brief An enumeration of library states
 * The shared library can be in one of the states defined in this enumeration.
 */
typedef enum
{
	/**
	 * \brief The library is in state uninitialized
	 * The library is not initialized. Either automounter_api_init() as never been called or automounter_api_deinit()
	 * has been called. Nearly all API member functions are not valid in this state and return RESULT_INVALID on a call.
	 */
	AM_API_UNINITIALIZED,

	/**
	 * \brief The library is in state initialized but not connected with the automounter
	 * The library is initialized but there is no connection to the automounter available by now. The application can
	 * use the API functions to register event callback functions. All API function that are requiring an active
	 * connection are not working in this state. Calling them will return RESULT_INVAID in this state.
	 */
	AM_API_INITIALIZED_DISCONNECTED,

	/**
	 * \brief The library wants to connect with the automounter and waits for it to appear.
	 * The library is waiting for the automounter daemon to come up.
	 * The shared library gets into this state on a call of automounter_api_try_connect() invoked while the
	 * automounter was not running. Once the automounter appears, the shared libraries tries to establish a connection
	 * automatically. Respective callback functions can be registered to get informed about the state changes.
	 * All member functions that are requiring an active connection are not working in this state. Calling them will
	 * return RESULT_INVALID.
	 */
	AM_API_INITIALIZED_WAITING_FOR_DAEMON,

	/**
	 * \brief The library is connecting with the automounter
	 * The library is trying to connect to the automounter in this state. All member functions that are requiring an
	 * active connection are not working. Calling them will return RESULT_INVALID.
	 */
	AM_API_INITIALIZED_CONNECTING,

	/**
	 * \brief The library is connected with the automounter
	 * The library is initialized and a connection to the automounter is established. The application can use
	 * the complete API exported by this shared library. The state can change at any time. Respective callback functions
	 * can be registered to get informed about state changes.
	 */
	AM_API_INITIALIZED_CONNECTED
} automounter_api_state_t;

/**
 * \brief Called to initialize the library.
 *
 * After a successful call, the library enters the state AM_API_INITIALIZED_DISCONNECTED. It does not try to connect to
 * the automounter at this point. If the library is not in state AM_API_UNINITIALIZED this call returns immediately
 * without any action.
 *
 * Important note: This function must not be called out of a callback function registered for an event with
 * the shared library!!!
 *
 * \param app_identifer  A string used to identify the application that uses the shared library in logs
 * \param loglevel  The log level, the shared library should log with
 * \param console_log_enabled  Logs a normally sent out to syslog. If this parameter is set true, logs are additionally
 * 			printed to the console.
 *
 * \return This functions returns RESULT_OK in case of no error. Following error codes might be returned in cased of
 * 		   failures:
 * 		   - RESULT_NORESOURCES: 			The function call failed due to resource problems.
 */
error_code_t automounter_api_init(const char *app_identifer, logger_loglevel_t loglevel, bool console_log_enabled);

/**
 * \brief Called to uninitialize the library.
 *
 * After a successful call, the library enters the state AM_API_UNINITIALIZED. A call to this function closes the
 * connection to the automounter if one is established. It cleans up all internally used resources. Registered event
 * callbacks are deregistered as well. The library reaches its initial state. If the library is already in state
 * AM_API_UNINITIALIZED, the function returns immediately without any action.
 *
 * Important note: This function must not be called out of a callback function registered for an event with
 * the shared library!!!
 */
void automounter_api_deinit(void);

/**
 * \brief Call to get the current state of the library.
 *
 * \return the current state of the library
 */
automounter_api_state_t automounter_api_get_state(void);

/**
 * \brief Call to get a file descriptor to detect that events are available for the library.
 *
 * The shared library receives events from the automounter but it does not fork any thread internally to process them.
 * So it needs a thread provided by the application for this. The application can detect incoming events by polling the
 * file descriptor returned by automounter_api_get_pollfd(). As soon as the application detects available events,
 * it simply needs to call automounter_api_dispatch_event() to allow the library to process the events. Applications can work
 * completely single threaded by integrating this mechanism into their main loop. Forking a new thread for processing
 * the automounter events is possible as well this way. It is important to know that any callback called by the library
 * is finally called with the thread that calls the automounter_api_dispatch_event() function.
 *
 * \return A file descriptor to detect events available for the library
 */
int automounter_api_get_pollfd(void);

/**
 * \brief Call to allow the library to process events if some are available.
 *
 * The shared library receives events from the automounter but it does not fork any thread internally to process them.
 * So it needs a thread provided by the application for this. The application can detect incoming events by polling the
 * file descriptor returned by automounter_api_get_pollfd(). As soon as the application detects available events,
 * it simply needs to call automounter_api_dispatch_event() to allow the library to process the events. Applications can work
 * completely single threaded by integrating this mechanism into their main loop. Forking a new thread for processing
 * the automounter events is possible as well this way. It is important to know that any callback called by the library
 * is finally called with the thread that calls the automounter_api_dispatch_event() function.
 *
 * Important note: This function must not be called out of a callback function registered for an event with
 * the shared library!!!
 *
 */
void automounter_api_dispatch_event(void);

/**
 * \brief Call to establish a connection to the automounter daemon immediately or once the daemon comes up.
 *
 * The shared library tries to establish a connection if the automounter daemon is already running. If it is not
 * running, the library enters the state AM_API_INITIALIZED_WAITING_FOR_DAEMON and waits for the daemon to appear.
 *
 * \return This functions returns RESULT_OK in case of no error. Following error codes might be returned in cased of
 * 		   failures:
 * 		   - RESULT_NORESOURCES: 			The function call failed due to resource problems.
 * 		   - RESULT_INVALID:				The library has not been initialized before
 * 		   - RESULT_CONNECT_TO_DAEMON_ERR:  The daemon is running but no connection could be establed for any reason
 * 		   									(in most cases the application has probably not the rights to access the
 * 		   									automounter socket)
 */
error_code_t automounter_api_try_connect(void);

/**
 * \brief Call to establish a connection to the automounter daemon immediately.
 *
 * The shared library tries to establish a connection. If the automounter is not running, this function returns
 * RESULT_DAEMON_NOT_RUNNING. If no connection could be established for any other reason
 * RESULT_CONNECT_TO_DAEMON_ERR is returned.
 *
* \return This functions returns RESULT_OK in case of no error. Following error codes might be returned in cased of
 * 		   failures:
 * 		   - RESULT_NORESOURCES: 			The function call failed due to resource problems.
 * 		   - RESULT_INVALID:				The library has not been initialized before
 * 		   - RESULT_DAEMON_NOT_RUNNING:		The daemon is not running
 * 		   - RESULT_CONNECT_TO_DAEMON_ERR:  The daemon is running but no connection could be established for any reason
 * 		   									(in most cases the application has probably not the rights to access the
 * 		   									automounter socket)
 */
error_code_t automounter_api_connect(void);

/**
 * \brief Call to disconnect from the automounter daemon.
 *
 * An established connection to the automounter is closed and the shared library enters the state
 * AM_API_INITIALIZED_DISCONNECTED. If the library is currently establishing a connecting or is waiting for the
 * automounter to appear, all ongoing activity is stopped and the library is brought into state
 * AM_API_INITIALIZED_DISCONNECTED as well. The function returns without any action if the shared library is in
 * state AM_API_UNINITIALIZED or in state AM_API_INITIALIZED_DISCONNECTED.
 */
void automounter_api_disconnect(void);

/**
 * \brief returns a string representing information about the version of the currently connected shared library.
 *
 * Application linking agaist this library shall provide a way to publish the string on demand for debugging reasons
 * via dlt, syslog, to the console, or any other way.
 *
 * \return a string representing version information about the currently connected shared library
 */
const char *automounter_api_get_version_string(void);

#ifdef __cplusplus
}
#endif

#endif
